/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.raft;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.Random;
import java.util.function.Supplier;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.network.ListenerName;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.raft.CandidateState;
import org.apache.kafka.raft.ElectionState;
import org.apache.kafka.raft.EpochState;
import org.apache.kafka.raft.FollowerState;
import org.apache.kafka.raft.LeaderAndEpoch;
import org.apache.kafka.raft.LeaderState;
import org.apache.kafka.raft.LogOffsetMetadata;
import org.apache.kafka.raft.OffsetAndEpoch;
import org.apache.kafka.raft.QuorumStateStore;
import org.apache.kafka.raft.ResignedState;
import org.apache.kafka.raft.UnattachedState;
import org.apache.kafka.raft.VotedState;
import org.apache.kafka.raft.internals.BatchAccumulator;
import org.apache.kafka.raft.internals.ReplicaKey;
import org.apache.kafka.raft.internals.VoterSet;
import org.slf4j.Logger;

public class QuorumState {
    private final OptionalInt localId;
    private final Uuid localDirectoryId;
    private final Time time;
    private final Logger log;
    private final QuorumStateStore store;
    private final ListenerName listenerName;
    private final Supplier<VoterSet> latestVoterSet;
    private final Supplier<Short> latestKraftVersion;
    private final Random random;
    private final int electionTimeoutMs;
    private final int fetchTimeoutMs;
    private final LogContext logContext;
    private volatile EpochState state;

    public QuorumState(OptionalInt localId, Uuid localDirectoryId, ListenerName listenerName, Supplier<VoterSet> latestVoterSet, Supplier<Short> latestKraftVersion, int electionTimeoutMs, int fetchTimeoutMs, QuorumStateStore store, Time time, LogContext logContext, Random random) {
        this.localId = localId;
        this.localDirectoryId = localDirectoryId;
        this.listenerName = listenerName;
        this.latestVoterSet = latestVoterSet;
        this.latestKraftVersion = latestKraftVersion;
        this.electionTimeoutMs = electionTimeoutMs;
        this.fetchTimeoutMs = fetchTimeoutMs;
        this.store = store;
        this.time = time;
        this.log = logContext.logger(QuorumState.class);
        this.random = random;
        this.logContext = logContext;
    }

    private ElectionState readElectionState() {
        ElectionState election = this.store.readElectionState().orElseGet(() -> ElectionState.withUnknownLeader(0, this.latestVoterSet.get().voterIds()));
        return election;
    }

    public void initialize(OffsetAndEpoch logEndOffsetAndEpoch) throws IllegalStateException {
        EpochState initialState;
        ElectionState election = this.readElectionState();
        if (election.hasVoted() && !this.localId.isPresent()) {
            throw new IllegalStateException(String.format("Initialized quorum state (%s) with a voted candidate but without a local id", election));
        }
        if (election.epoch() < logEndOffsetAndEpoch.epoch()) {
            this.log.warn("Epoch from quorum store file ({}) is {}, which is smaller than last written epoch {} in the log", new Object[]{this.store.path(), election.epoch(), logEndOffsetAndEpoch.epoch()});
            initialState = new UnattachedState(this.time, logEndOffsetAndEpoch.epoch(), this.latestVoterSet.get().voterIds(), Optional.empty(), this.randomElectionTimeoutMs(), this.logContext);
        } else if (this.localId.isPresent() && election.isLeader(this.localId.getAsInt())) {
            initialState = new ResignedState(this.time, this.localId.getAsInt(), election.epoch(), this.latestVoterSet.get().voterIds(), this.randomElectionTimeoutMs(), Collections.emptyList(), this.logContext);
        } else if (this.localId.isPresent() && election.isVotedCandidate(ReplicaKey.of(this.localId.getAsInt(), Optional.of(this.localDirectoryId)))) {
            initialState = new CandidateState(this.time, this.localId.getAsInt(), this.localDirectoryId, election.epoch(), this.latestVoterSet.get(), Optional.empty(), 1, this.randomElectionTimeoutMs(), this.logContext);
        } else if (election.hasVoted()) {
            initialState = new VotedState(this.time, election.epoch(), election.votedKey(), this.latestVoterSet.get().voterIds(), Optional.empty(), this.randomElectionTimeoutMs(), this.logContext);
        } else if (election.hasLeader()) {
            Node leader = this.latestVoterSet.get().voterNode(election.leaderId(), this.listenerName).orElseThrow(() -> new IllegalStateException(String.format("Leader %s must be in the voter set %s", election.leaderId(), this.latestVoterSet.get())));
            initialState = new FollowerState(this.time, election.epoch(), leader, this.latestVoterSet.get().voterIds(), Optional.empty(), this.fetchTimeoutMs, this.logContext);
        } else {
            initialState = new UnattachedState(this.time, election.epoch(), this.latestVoterSet.get().voterIds(), Optional.empty(), this.randomElectionTimeoutMs(), this.logContext);
        }
        this.durableTransitionTo(initialState);
    }

    public boolean isOnlyVoter() {
        return this.localId.isPresent() && this.latestVoterSet.get().isOnlyVoter(ReplicaKey.of(this.localId.getAsInt(), Optional.of(this.localDirectoryId)));
    }

    public int localIdOrSentinel() {
        return this.localId.orElse(-1);
    }

    public int localIdOrThrow() {
        return this.localId.orElseThrow(() -> new IllegalStateException("Required local id is not present"));
    }

    public OptionalInt localId() {
        return this.localId;
    }

    public Uuid localDirectoryId() {
        return this.localDirectoryId;
    }

    public int epoch() {
        return this.state.epoch();
    }

    public int leaderIdOrSentinel() {
        return this.state.election().leaderIdOrSentinel();
    }

    public Optional<LogOffsetMetadata> highWatermark() {
        return this.state.highWatermark();
    }

    public OptionalInt leaderId() {
        ElectionState election = this.state.election();
        if (election.hasLeader()) {
            return OptionalInt.of(this.state.election().leaderId());
        }
        return OptionalInt.empty();
    }

    public boolean hasLeader() {
        return this.leaderId().isPresent();
    }

    public boolean hasRemoteLeader() {
        return this.hasLeader() && this.leaderIdOrSentinel() != this.localIdOrSentinel();
    }

    public boolean isVoter() {
        if (!this.localId.isPresent()) {
            return false;
        }
        return this.latestVoterSet.get().isVoter(ReplicaKey.of(this.localId.getAsInt(), Optional.of(this.localDirectoryId)));
    }

    public boolean isVoter(ReplicaKey nodeKey) {
        return this.latestVoterSet.get().isVoter(nodeKey);
    }

    public boolean isObserver() {
        return !this.isVoter();
    }

    public void transitionToResigned(List<Integer> preferredSuccessors) {
        if (!this.isLeader()) {
            throw new IllegalStateException("Invalid transition to Resigned state from " + this.state);
        }
        int epoch = this.state.epoch();
        this.memoryTransitionTo(new ResignedState(this.time, this.localIdOrThrow(), epoch, this.latestVoterSet.get().voterIds(), this.randomElectionTimeoutMs(), preferredSuccessors, this.logContext));
    }

    public void transitionToUnattached(int epoch) {
        int currentEpoch = this.state.epoch();
        if (epoch <= currentEpoch) {
            throw new IllegalStateException("Cannot transition to Unattached with epoch= " + epoch + " from current state " + this.state);
        }
        long electionTimeoutMs = this.isObserver() ? Long.MAX_VALUE : (this.isCandidate() ? this.candidateStateOrThrow().remainingElectionTimeMs(this.time.milliseconds()) : (this.isVoted() ? this.votedStateOrThrow().remainingElectionTimeMs(this.time.milliseconds()) : (this.isUnattached() ? this.unattachedStateOrThrow().remainingElectionTimeMs(this.time.milliseconds()) : (long)this.randomElectionTimeoutMs())));
        this.durableTransitionTo(new UnattachedState(this.time, epoch, this.latestVoterSet.get().voterIds(), this.state.highWatermark(), electionTimeoutMs, this.logContext));
    }

    public void transitionToVoted(int epoch, ReplicaKey candidateKey) {
        int currentEpoch = this.state.epoch();
        if (this.localId.isPresent() && candidateKey.id() == this.localId.getAsInt()) {
            throw new IllegalStateException(String.format("Cannot transition to Voted for %s and epoch %d since it matches the local broker.id", candidateKey, epoch));
        }
        if (!this.localId.isPresent()) {
            throw new IllegalStateException("Cannot transition to voted without a replica id");
        }
        if (epoch < currentEpoch) {
            throw new IllegalStateException(String.format("Cannot transition to Voted for %s and epoch %d since the current epoch (%d) is larger", candidateKey, epoch, currentEpoch));
        }
        if (epoch == currentEpoch && !this.isUnattached()) {
            throw new IllegalStateException(String.format("Cannot transition to Voted for %s and epoch %d from the current state (%s)", candidateKey, epoch, this.state));
        }
        this.durableTransitionTo(new VotedState(this.time, epoch, candidateKey, this.latestVoterSet.get().voterIds(), this.state.highWatermark(), this.randomElectionTimeoutMs(), this.logContext));
    }

    public void transitionToFollower(int epoch, Node leader) {
        int currentEpoch = this.state.epoch();
        if (this.localId.isPresent() && leader.id() == this.localId.getAsInt()) {
            throw new IllegalStateException("Cannot transition to Follower with leader " + leader + " and epoch " + epoch + " since it matches the local broker.id " + this.localId);
        }
        if (epoch < currentEpoch) {
            throw new IllegalStateException("Cannot transition to Follower with leader " + leader + " and epoch " + epoch + " since the current epoch " + currentEpoch + " is larger");
        }
        if (epoch == currentEpoch && (this.isFollower() || this.isLeader())) {
            throw new IllegalStateException("Cannot transition to Follower with leader " + leader + " and epoch " + epoch + " from state " + this.state);
        }
        this.durableTransitionTo(new FollowerState(this.time, epoch, leader, this.latestVoterSet.get().voterIds(), this.state.highWatermark(), this.fetchTimeoutMs, this.logContext));
    }

    public void transitionToCandidate() {
        if (this.isObserver()) {
            throw new IllegalStateException(String.format("Cannot transition to Candidate since the local id (%s) and directory id (%s) is not one of the voters %s", this.localId, this.localDirectoryId, this.latestVoterSet.get()));
        }
        if (this.isLeader()) {
            throw new IllegalStateException("Cannot transition to Candidate since the local broker.id=" + this.localId + " since this node is already a Leader with state " + this.state);
        }
        int retries = this.isCandidate() ? this.candidateStateOrThrow().retries() + 1 : 1;
        int newEpoch = this.epoch() + 1;
        int electionTimeoutMs = this.randomElectionTimeoutMs();
        this.durableTransitionTo(new CandidateState(this.time, this.localIdOrThrow(), this.localDirectoryId, newEpoch, this.latestVoterSet.get(), this.state.highWatermark(), retries, electionTimeoutMs, this.logContext));
    }

    public <T> LeaderState<T> transitionToLeader(long epochStartOffset, BatchAccumulator<T> accumulator) {
        if (this.isObserver()) {
            throw new IllegalStateException(String.format("Cannot transition to Leader since the local id (%s) and directory id (%s) is not one of the voters %s", this.localId, this.localDirectoryId, this.latestVoterSet.get()));
        }
        if (!this.isCandidate()) {
            throw new IllegalStateException("Cannot transition to Leader from current state " + this.state);
        }
        CandidateState candidateState = this.candidateStateOrThrow();
        if (!candidateState.isVoteGranted()) {
            throw new IllegalStateException("Cannot become leader without majority votes granted");
        }
        LeaderState<T> state = new LeaderState<T>(this.time, this.localIdOrThrow(), this.epoch(), epochStartOffset, this.latestVoterSet.get().voterIds(), candidateState.grantingVoters(), accumulator, this.fetchTimeoutMs, this.logContext);
        this.durableTransitionTo(state);
        return state;
    }

    private void durableTransitionTo(EpochState newState) {
        if (this.state != null) {
            try {
                this.state.close();
            }
            catch (IOException e) {
                throw new UncheckedIOException("Failed to transition from " + this.state.name() + " to " + newState.name(), e);
            }
        }
        this.store.writeElectionState(newState.election(), this.latestKraftVersion.get());
        this.memoryTransitionTo(newState);
    }

    private void memoryTransitionTo(EpochState newState) {
        EpochState from = this.state;
        this.state = newState;
        this.log.info("Completed transition to {} from {}", (Object)newState, (Object)from);
    }

    private int randomElectionTimeoutMs() {
        if (this.electionTimeoutMs == 0) {
            return 0;
        }
        return this.electionTimeoutMs + this.random.nextInt(this.electionTimeoutMs);
    }

    public boolean canGrantVote(ReplicaKey candidateKey, boolean isLogUpToDate) {
        return this.state.canGrantVote(candidateKey, isLogUpToDate);
    }

    public FollowerState followerStateOrThrow() {
        if (this.isFollower()) {
            return (FollowerState)this.state;
        }
        throw new IllegalStateException("Expected to be Follower, but the current state is " + this.state);
    }

    public VotedState votedStateOrThrow() {
        return this.maybeVotedState().orElseThrow(() -> new IllegalStateException("Expected to be Voted, but current state is " + this.state));
    }

    public Optional<VotedState> maybeVotedState() {
        EpochState fixedState = this.state;
        if (fixedState instanceof VotedState) {
            return Optional.of((VotedState)fixedState);
        }
        return Optional.empty();
    }

    public UnattachedState unattachedStateOrThrow() {
        if (this.isUnattached()) {
            return (UnattachedState)this.state;
        }
        throw new IllegalStateException("Expected to be Unattached, but current state is " + this.state);
    }

    public <T> LeaderState<T> leaderStateOrThrow() {
        return this.maybeLeaderState().orElseThrow(() -> new IllegalStateException("Expected to be Leader, but current state is " + this.state));
    }

    public <T> Optional<LeaderState<T>> maybeLeaderState() {
        EpochState fixedState = this.state;
        if (fixedState instanceof LeaderState) {
            return Optional.of((LeaderState)fixedState);
        }
        return Optional.empty();
    }

    public ResignedState resignedStateOrThrow() {
        if (this.isResigned()) {
            return (ResignedState)this.state;
        }
        throw new IllegalStateException("Expected to be Resigned, but current state is " + this.state);
    }

    public CandidateState candidateStateOrThrow() {
        if (this.isCandidate()) {
            return (CandidateState)this.state;
        }
        throw new IllegalStateException("Expected to be Candidate, but current state is " + this.state);
    }

    public LeaderAndEpoch leaderAndEpoch() {
        ElectionState election = this.state.election();
        return new LeaderAndEpoch(election.optionalLeaderId(), election.epoch());
    }

    public boolean isFollower() {
        return this.state instanceof FollowerState;
    }

    public boolean isVoted() {
        return this.state instanceof VotedState;
    }

    public boolean isUnattached() {
        return this.state instanceof UnattachedState;
    }

    public boolean isLeader() {
        return this.state instanceof LeaderState;
    }

    public boolean isResigned() {
        return this.state instanceof ResignedState;
    }

    public boolean isCandidate() {
        return this.state instanceof CandidateState;
    }
}

